-- |
-- Binary Communicator
-- 
-- This module provides the datatype BinaryCom, which enables you
-- to easily send and receive data to and from a binary source.
-- The transmitted data can be an instance of the 'Binary' class,
-- or you can provide your own Put and Get actions to serialize
-- and parse the binary stream.

module Data.BinaryCom
  (BinaryCom, BinaryTrt,
   binaryCom, binaryComTrt, binaryCom2H, binaryComBS,
   send, flush, sendFlush, receive,
   sendPut, receiveGet)
where

import System.IO
import Data.IORef
import Control.Monad.Trans
import qualified Data.ByteString.Lazy as L
import qualified Data.Binary as B
import qualified Data.Binary.Get as B
import qualified Data.Binary.Put as B


-- | 'BinaryCom' type
data BinaryCom = BinaryCom (IORef L.ByteString) -- For reading
                           Handle               -- For writing
                           BinaryTrt            -- For sent data

-- | Function that applies a treatment (e.g. compression) to binary data
type BinaryTrt = L.ByteString -> L.ByteString

-- | Creates a 'BinaryCom' from a 'Handle' opened for both reading and writing.
-- Be careful not to use the handle afterwards
binaryCom :: (MonadIO m) => Handle -> m BinaryCom
binaryCom h = binaryComTrt h id id

-- | Creates a 'BinaryCom' which will apply binary treatments (e.g. compression or decompression) to received and sent data
binaryComTrt :: (MonadIO m) =>
                Handle      -- ^ For reading and writing
             -> BinaryTrt   -- ^ For received data
             -> BinaryTrt   -- ^ For sent data
             -> m BinaryCom -- ^ New 'BinaryCom'
binaryComTrt h trtR trtW = binaryCom2H h trtR h trtW

-- | Creates a 'BinaryCom' from two 'Handle's: one for reading, one for writing
binaryCom2H :: (MonadIO m) =>
               Handle      -- ^ For reading
            -> BinaryTrt   -- ^ For received data
            -> Handle      -- ^ For writing
            -> BinaryTrt   -- ^ For sent data
            -> m BinaryCom -- ^ New 'BinaryCom'
binaryCom2H hR trtR hW trtW = do
  inp <- liftIO $ L.hGetContents hR
  binaryComBS (trtR inp) hW trtW

-- | Creates a 'BinaryCom' from a lazy 'L.ByteString' (for reading) and a 'Handle' (for writing)
binaryComBS :: (MonadIO m) =>
               L.ByteString -- ^ For reading
            -> Handle       -- ^ For writing
            -> BinaryTrt    -- ^ For sent data
            -> m BinaryCom  -- ^ New 'BinaryCom'
binaryComBS inp hW trtW = liftIO $ do
  ref <- newIORef inp
  return $ BinaryCom ref hW trtW

-- | Sends a serializable value through a 'BinaryCom'
send :: (B.Binary a, MonadIO m) => BinaryCom -> a -> m ()
send b = sendPut b . B.put

-- | Flushes a 'BinaryCom'. Do not forget to do this after sending!
flush :: (MonadIO m) => BinaryCom -> m ()
flush (BinaryCom _ hW _) = liftIO $ hFlush hW

-- | Shortcut for sending a value and flushing the 'BinaryCom'
sendFlush :: (B.Binary a, MonadIO m) => BinaryCom -> a -> m ()
sendFlush b v = send b v >> flush b

-- | Receives a serializable value through a 'BinaryCom'
receive :: (B.Binary a, MonadIO m) => BinaryCom -> m a
receive b = receiveGet b B.get

-- | Runs a 'B.Put' monad and sends its result
sendPut :: (MonadIO m) => BinaryCom -> B.Put -> m ()
sendPut (BinaryCom _ hW trtW) putAction = liftIO $
  L.hPut hW . trtW $ B.runPut putAction

-- | Receives a value. Runs a 'B.Get' monad to parse it
receiveGet :: (MonadIO m) => BinaryCom -> B.Get a -> m a
receiveGet (BinaryCom ref _ _) getAction = liftIO $ do
  inp <- readIORef ref
  let (val, inpRest, _) = B.runGetState getAction inp 0
  writeIORef ref inpRest
  return val

