# 1 "Shaders.hs_pre"
# 1 "<built-in>"
# 1 "<command-line>"
# 1 "Shaders.hs_pre"
--------------------------------------------------------------------------------
-- |
-- Module : Graphics.Rendering.OpenGL.GL.Shaders
-- Copyright : (c) Sven Panne 2002-2006
-- License : BSD-style (see the file libraries/OpenGL/LICENSE)
--
-- Maintainer : sven.panne@aedion.de
-- Stability : stable
-- Portability : portable
--
-- This module corresponds to sections 2.15 (Vertex Shaders) and section 3.11
-- (Fragment Shaders) of the OpenGL 2.1 specs.
--
--------------------------------------------------------------------------------

module Graphics.Rendering.OpenGL.GL.Shaders (
   -- * Shader Objects
   Shader, VertexShader, FragmentShader, shaderDeleteStatus, shaderSource,
   compileShader, compileStatus, shaderInfoLog,

   -- * Program Objects
   Program, programDeleteStatus, attachedShaders, linkProgram, linkStatus,
   programInfoLog, validateProgram, validateStatus, currentProgram,

   -- * Vertex attributes
   attribLocation, VariableType(..), activeAttribs,

   -- * Uniform variables
   UniformLocation, uniformLocation, activeUniforms, Uniform(..),
   UniformComponent,

   -- * Implementation limits related to GLSL
   maxVertexTextureImageUnits, maxTextureImageUnits,
   maxCombinedTextureImageUnits, maxTextureCoords, maxVertexUniformComponents,
   maxFragmentUniformComponents, maxVertexAttribs, maxVaryingFloats
) where

import Control.Monad ( replicateM, mapM_, foldM )
import Control.Monad.Fix ( MonadFix(..) )
import Data.Int
import Data.List ( genericLength, (\\) )
import Foreign.C.String ( peekCAStringLen, withCAStringLen )
import Foreign.Marshal.Alloc ( alloca, allocaBytes )
import Foreign.Marshal.Array ( allocaArray, withArray, peekArray )
import Foreign.Marshal.Utils ( withMany )
import Foreign.Ptr ( Ptr, castPtr, nullPtr )
import Foreign.Storable ( Storable(peek,sizeOf) )
import Graphics.Rendering.OpenGL.GL.BasicTypes (
   GLboolean, GLchar, GLint, GLuint, GLsizei, GLenum, GLfloat )
import Graphics.Rendering.OpenGL.GL.BufferObjects ( ObjectName(..) )
import Graphics.Rendering.OpenGL.GL.Extensions (
   FunPtr, unsafePerformIO, Invoker, getProcAddress )
import Graphics.Rendering.OpenGL.GL.GLboolean ( unmarshalGLboolean )
import Graphics.Rendering.OpenGL.GL.PeekPoke ( peek1 )
import Graphics.Rendering.OpenGL.GL.QueryUtils (
   GetPName(GetMaxCombinedTextureImageUnits, GetMaxFragmentUniformComponents,
            GetMaxTextureCoords, GetMaxTextureImageUnits,GetMaxVaryingFloats,
            GetMaxVertexAttribs, GetMaxVertexTextureImageUnits,
            GetMaxVertexUniformComponents, GetCurrentProgram),
   getInteger1, getSizei1 )
import Graphics.Rendering.OpenGL.GL.StateVar (
   HasGetter(get), GettableStateVar, makeGettableStateVar, StateVar,
   makeStateVar )
import Graphics.Rendering.OpenGL.GL.VertexSpec (
   AttribLocation(..), Vertex2(..), Vertex3(..), Vertex4(..), TexCoord1(..),
   TexCoord2(..), TexCoord3(..), TexCoord4(..), Normal3(..), FogCoord1(..),
   Color3(..), Color4(..), Index1(..) )

--------------------------------------------------------------------------------

# 1 "../../../../include/HsOpenGLExt.h" 1
# 72 "Shaders.hs_pre" 2
# 1 "../../../../include/HsOpenGLTypes.h" 1
# 18 "../../../../include/HsOpenGLTypes.h"
# 1 "../../../../include/HsOpenGLConfig.h" 1
# 19 "../../../../include/HsOpenGLTypes.h" 2
# 73 "Shaders.hs_pre" 2

--------------------------------------------------------------------------------

type GLStringLen = (Ptr GLchar, GLsizei)

peekGLstringLen :: GLStringLen -> IO String
peekGLstringLen (p,l) = peekCAStringLen (castPtr p, fromIntegral l)

withGLStringLen :: String -> (GLStringLen -> IO a) -> IO a
withGLStringLen s act =
   withCAStringLen s $ \(p,len) ->
      act (castPtr p, fromIntegral len)

--------------------------------------------------------------------------------

newtype VertexShader = VertexShader { vertexShaderID :: GLuint }
   deriving ( Eq, Ord, Show )

newtype FragmentShader = FragmentShader { fragmentShaderID :: GLuint }
   deriving ( Eq, Ord, Show )

--------------------------------------------------------------------------------

class (Eq s, Ord s, Show s, ObjectName s) => Shader s where
   shaderID :: s -> GLuint
   makeShader :: GLuint -> s
   shaderType :: s -> GLenum

instance Shader VertexShader where
   makeShader = VertexShader
   shaderID = vertexShaderID
   shaderType = const 0x8B31

instance Shader FragmentShader where
   makeShader = FragmentShader
   shaderID = fragmentShaderID
   shaderType = const 0x8B30

--------------------------------------------------------------------------------

instance ObjectName VertexShader where
   genObjectNames = genShaderNames
   deleteObjectNames = deleteShaderNames
   isObjectName = isShaderName

instance ObjectName FragmentShader where
   genObjectNames = genShaderNames
   deleteObjectNames = deleteShaderNames
   isObjectName = isShaderName

genShaderNames :: Shader s => Int -> IO [s]
genShaderNames n = replicateM n createShader

createShader :: Shader s => IO s
createShader = mfix (fmap makeShader . glCreateShader . shaderType)

deleteShaderNames :: Shader s => [s] -> IO ()
deleteShaderNames = mapM_ (glDeleteShader . shaderID)

isShaderName :: Shader s => s -> IO Bool
isShaderName = fmap unmarshalGLboolean . glIsShader . shaderID

foreign import unsafe "dynamic" dyn_glCreateShader :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (GLenum -> IO GLuint) ; glCreateShader :: (GLenum -> IO GLuint) ; glCreateShader = dyn_glCreateShader ptr_glCreateShader ; ptr_glCreateShader :: FunPtr a ; ptr_glCreateShader = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glDeleteShader :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (GLuint -> IO ()) ; glDeleteShader :: (GLuint -> IO ()) ; glDeleteShader = dyn_glDeleteShader ptr_glDeleteShader ; ptr_glDeleteShader :: FunPtr a ; ptr_glDeleteShader = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glIsShader :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (GLuint -> IO GLboolean) ; glIsShader :: (GLuint -> IO GLboolean) ; glIsShader = dyn_glIsShader ptr_glIsShader ; ptr_glIsShader :: FunPtr a ; ptr_glIsShader = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

--------------------------------------------------------------------------------

compileShader :: Shader s => s -> IO ()
compileShader = glCompileShader . shaderID

foreign import unsafe "dynamic" dyn_glCompileShader :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (GLuint -> IO ()) ; glCompileShader :: (GLuint -> IO ()) ; glCompileShader = dyn_glCompileShader ptr_glCompileShader ; ptr_glCompileShader :: FunPtr a ; ptr_glCompileShader = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

--------------------------------------------------------------------------------

shaderSource :: Shader s => s -> StateVar [String]
shaderSource shader =
   makeStateVar (getShaderSource shader) (setShaderSource shader)

setShaderSource :: Shader s => s -> [String] -> IO ()
setShaderSource shader srcs = do
   let len = genericLength srcs
   withMany withGLStringLen srcs $ \charBufsAndLengths -> do
      let (charBufs, lengths) = unzip charBufsAndLengths
      withArray charBufs $ \charBufsBuf ->
         withArray lengths $ \lengthsBuf ->
            glShaderSource (shaderID shader) len charBufsBuf lengthsBuf

foreign import unsafe "dynamic" dyn_glShaderSource :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (GLuint -> GLsizei -> Ptr (Ptr GLchar) -> Ptr GLint -> IO ()) ; glShaderSource :: (GLuint -> GLsizei -> Ptr (Ptr GLchar) -> Ptr GLint -> IO ()) ; glShaderSource = dyn_glShaderSource ptr_glShaderSource ; ptr_glShaderSource :: FunPtr a ; ptr_glShaderSource = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

getShaderSource :: Shader s => s -> IO [String]
getShaderSource shader = do
   src <- get (stringQuery (shaderSourceLength shader)
                           (glGetShaderSource (shaderID shader)))
   return [src]

foreign import unsafe "dynamic" dyn_glGetShaderSource :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (GLuint -> GLsizei -> Ptr GLsizei -> Ptr GLchar -> IO ()) ; glGetShaderSource :: (GLuint -> GLsizei -> Ptr GLsizei -> Ptr GLchar -> IO ()) ; glGetShaderSource = dyn_glGetShaderSource ptr_glGetShaderSource ; ptr_glGetShaderSource :: FunPtr a ; ptr_glGetShaderSource = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

stringQuery :: GettableStateVar GLsizei -> (GLsizei -> Ptr GLsizei -> Ptr GLchar -> IO ()) -> GettableStateVar String
stringQuery lengthVar getStr =
   makeGettableStateVar $ do
      len <- get lengthVar -- Note: This includes the NUL character!
      if len == 0
        then return ""
        else allocaArray (fromIntegral len) $ \buf -> do
                getStr len nullPtr buf
                peekGLstringLen (buf, len-1)

--------------------------------------------------------------------------------

shaderInfoLog :: Shader s => s -> GettableStateVar String
shaderInfoLog shader =
   stringQuery (shaderInfoLogLength shader) (glGetShaderInfoLog (shaderID shader))

foreign import unsafe "dynamic" dyn_glGetShaderInfoLog :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (GLuint -> GLsizei -> Ptr GLsizei -> Ptr GLchar -> IO ()) ; glGetShaderInfoLog :: (GLuint -> GLsizei -> Ptr GLsizei -> Ptr GLchar -> IO ()) ; glGetShaderInfoLog = dyn_glGetShaderInfoLog ptr_glGetShaderInfoLog ; ptr_glGetShaderInfoLog :: FunPtr a ; ptr_glGetShaderInfoLog = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

--------------------------------------------------------------------------------

shaderDeleteStatus :: Shader s => s -> GettableStateVar Bool
shaderDeleteStatus = shaderVar unmarshalGLboolean ShaderDeleteStatus

compileStatus :: Shader s => s -> GettableStateVar Bool
compileStatus = shaderVar unmarshalGLboolean CompileStatus

shaderInfoLogLength :: Shader s => s -> GettableStateVar GLsizei
shaderInfoLogLength = shaderVar fromIntegral ShaderInfoLogLength

shaderSourceLength :: Shader s => s -> GettableStateVar GLsizei
shaderSourceLength = shaderVar fromIntegral ShaderSourceLength

shaderTypeEnum :: Shader s => s -> GettableStateVar GLenum
shaderTypeEnum = shaderVar fromIntegral ShaderType

--------------------------------------------------------------------------------

data GetShaderPName =
     ShaderDeleteStatus
   | CompileStatus
   | ShaderInfoLogLength
   | ShaderSourceLength
   | ShaderType

marshalGetShaderPName :: GetShaderPName -> GLenum
marshalGetShaderPName x = case x of
   ShaderDeleteStatus -> 0x8B80
   CompileStatus -> 0x8B81
   ShaderInfoLogLength -> 0x8B84
   ShaderSourceLength -> 0x8B88
   ShaderType -> 0x8B4F

shaderVar :: Shader s => (GLint -> a) -> GetShaderPName -> s -> GettableStateVar a
shaderVar f p shader =
   makeGettableStateVar $
      alloca $ \buf -> do
         glGetShaderiv (shaderID shader) (marshalGetShaderPName p) buf
         peek1 f buf

foreign import unsafe "dynamic" dyn_glGetShaderiv :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (GLuint -> GLenum -> Ptr GLint -> IO ()) ; glGetShaderiv :: (GLuint -> GLenum -> Ptr GLint -> IO ()) ; glGetShaderiv = dyn_glGetShaderiv ptr_glGetShaderiv ; ptr_glGetShaderiv :: FunPtr a ; ptr_glGetShaderiv = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

--------------------------------------------------------------------------------

newtype Program = Program { programID :: GLuint }
   deriving ( Eq, Ord, Show )

instance ObjectName Program where
   genObjectNames n = replicateM n $ fmap Program glCreateProgram
   deleteObjectNames = mapM_ (glDeleteProgram . programID)
   isObjectName = fmap unmarshalGLboolean . glIsProgram . programID

foreign import unsafe "dynamic" dyn_glCreateProgram :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (IO GLuint) ; glCreateProgram :: (IO GLuint) ; glCreateProgram = dyn_glCreateProgram ptr_glCreateProgram ; ptr_glCreateProgram :: FunPtr a ; ptr_glCreateProgram = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glDeleteProgram :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (GLuint -> IO ()) ; glDeleteProgram :: (GLuint -> IO ()) ; glDeleteProgram = dyn_glDeleteProgram ptr_glDeleteProgram ; ptr_glDeleteProgram :: FunPtr a ; ptr_glDeleteProgram = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glIsProgram :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (GLuint -> IO GLboolean) ; glIsProgram :: (GLuint -> IO GLboolean) ; glIsProgram = dyn_glIsProgram ptr_glIsProgram ; ptr_glIsProgram :: FunPtr a ; ptr_glIsProgram = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

--------------------------------------------------------------------------------

attachedShaders :: Program -> StateVar ([VertexShader],[FragmentShader])
attachedShaders program =
   makeStateVar (getAttachedShaders program) (setAttachedShaders program)

getAttachedShaders :: Program -> IO ([VertexShader],[FragmentShader])
getAttachedShaders program = getAttachedShaderIDs program >>= splitShaderIDs

getAttachedShaderIDs :: Program -> IO [GLuint]
getAttachedShaderIDs program = do
   numShaders <- get (numAttachedShaders program)
   allocaArray (fromIntegral numShaders) $ \buf -> do
      glGetAttachedShaders (programID program) numShaders nullPtr buf
      peekArray (fromIntegral numShaders) buf

foreign import unsafe "dynamic" dyn_glGetAttachedShaders :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (GLuint -> GLsizei -> Ptr GLsizei -> Ptr GLuint -> IO ()) ; glGetAttachedShaders :: (GLuint -> GLsizei -> Ptr GLsizei -> Ptr GLuint -> IO ()) ; glGetAttachedShaders = dyn_glGetAttachedShaders ptr_glGetAttachedShaders ; ptr_glGetAttachedShaders :: FunPtr a ; ptr_glGetAttachedShaders = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

splitShaderIDs :: [GLuint] -> IO ([VertexShader],[FragmentShader])
splitShaderIDs ids = do
   (vs, fs) <- partitionM isVertexShaderID ids
   return (map VertexShader vs, map FragmentShader fs)

isVertexShaderID :: GLuint -> IO Bool
isVertexShaderID x = do
   t <- get (shaderTypeEnum (VertexShader x))
   return $ t == shaderType (undefined :: VertexShader)

partitionM :: (a -> IO Bool) -> [a] -> IO ([a],[a])
partitionM p = foldM select ([],[])
   where select (ts, fs) x = do
            b <- p x
            return $ if b then (x:ts, fs) else (ts, x:fs)

setAttachedShaders :: Program -> ([VertexShader],[FragmentShader]) -> IO ()
setAttachedShaders program (vs, fs) = do
   currentIDs <- getAttachedShaderIDs program
   let newIDs = map shaderID vs ++ map shaderID fs
   mapM_ (glAttachShader program) (newIDs \\ currentIDs)
   mapM_ (glDetachShader program) (currentIDs \\ newIDs)

foreign import unsafe "dynamic" dyn_glAttachShader :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (Program -> GLuint -> IO ()) ; glAttachShader :: (Program -> GLuint -> IO ()) ; glAttachShader = dyn_glAttachShader ptr_glAttachShader ; ptr_glAttachShader :: FunPtr a ; ptr_glAttachShader = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glDetachShader :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (Program -> GLuint -> IO ()) ; glDetachShader :: (Program -> GLuint -> IO ()) ; glDetachShader = dyn_glDetachShader ptr_glDetachShader ; ptr_glDetachShader :: FunPtr a ; ptr_glDetachShader = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

--------------------------------------------------------------------------------

linkProgram :: Program -> IO ()
linkProgram = glLinkProgram

foreign import unsafe "dynamic" dyn_glLinkProgram :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (Program -> IO ()) ; glLinkProgram :: (Program -> IO ()) ; glLinkProgram = dyn_glLinkProgram ptr_glLinkProgram ; ptr_glLinkProgram :: FunPtr a ; ptr_glLinkProgram = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

currentProgram :: StateVar (Maybe Program)
currentProgram =
   makeStateVar
      (do p <- getCurrentProgram
          return $ if p == noProgram then Nothing else Just p)
      (glUseProgram . maybe noProgram id)

getCurrentProgram :: IO Program
getCurrentProgram = fmap Program $ getInteger1 fromIntegral GetCurrentProgram

noProgram :: Program
noProgram = Program 0

foreign import unsafe "dynamic" dyn_glUseProgram :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (Program -> IO ()) ; glUseProgram :: (Program -> IO ()) ; glUseProgram = dyn_glUseProgram ptr_glUseProgram ; ptr_glUseProgram :: FunPtr a ; ptr_glUseProgram = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

validateProgram :: Program -> IO ()
validateProgram = glValidateProgram

foreign import unsafe "dynamic" dyn_glValidateProgram :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (Program -> IO ()) ; glValidateProgram :: (Program -> IO ()) ; glValidateProgram = dyn_glValidateProgram ptr_glValidateProgram ; ptr_glValidateProgram :: FunPtr a ; ptr_glValidateProgram = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

programInfoLog :: Program -> GettableStateVar String
programInfoLog p =
   stringQuery (programInfoLogLength p) (glGetProgramInfoLog (programID p))

foreign import unsafe "dynamic" dyn_glGetProgramInfoLog :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (GLuint -> GLsizei -> Ptr GLsizei -> Ptr GLchar -> IO ()) ; glGetProgramInfoLog :: (GLuint -> GLsizei -> Ptr GLsizei -> Ptr GLchar -> IO ()) ; glGetProgramInfoLog = dyn_glGetProgramInfoLog ptr_glGetProgramInfoLog ; ptr_glGetProgramInfoLog :: FunPtr a ; ptr_glGetProgramInfoLog = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

--------------------------------------------------------------------------------

programDeleteStatus :: Program -> GettableStateVar Bool
programDeleteStatus = programVar unmarshalGLboolean ProgramDeleteStatus

linkStatus :: Program -> GettableStateVar Bool
linkStatus = programVar unmarshalGLboolean LinkStatus

validateStatus :: Program -> GettableStateVar Bool
validateStatus = programVar unmarshalGLboolean ValidateStatus

programInfoLogLength :: Program -> GettableStateVar GLsizei
programInfoLogLength = programVar fromIntegral ProgramInfoLogLength

numAttachedShaders :: Program -> GettableStateVar GLsizei
numAttachedShaders = programVar fromIntegral AttachedShaders

activeAttributes :: Program -> GettableStateVar GLuint
activeAttributes = programVar fromIntegral ActiveAttributes

activeAttributeMaxLength :: Program -> GettableStateVar GLsizei
activeAttributeMaxLength = programVar fromIntegral ActiveAttributeMaxLength

numActiveUniforms :: Program -> GettableStateVar GLuint
numActiveUniforms = programVar fromIntegral ActiveUniforms

activeUniformMaxLength :: Program -> GettableStateVar GLsizei
activeUniformMaxLength = programVar fromIntegral ActiveUniformMaxLength

--------------------------------------------------------------------------------

data GetProgramPName =
     ProgramDeleteStatus
   | LinkStatus
   | ValidateStatus
   | ProgramInfoLogLength
   | AttachedShaders
   | ActiveAttributes
   | ActiveAttributeMaxLength
   | ActiveUniforms
   | ActiveUniformMaxLength

marshalGetProgramPName :: GetProgramPName -> GLenum
marshalGetProgramPName x = case x of
   ProgramDeleteStatus -> 0x8B80
   LinkStatus -> 0x8B82
   ValidateStatus -> 0x8B83
   ProgramInfoLogLength -> 0x8B84
   AttachedShaders -> 0x8B85
   ActiveAttributes -> 0x8B89
   ActiveAttributeMaxLength -> 0x8B8A
   ActiveUniforms -> 0x8B86
   ActiveUniformMaxLength -> 0x8B87

programVar :: (GLint -> a) -> GetProgramPName -> Program -> GettableStateVar a
programVar f p program =
   makeGettableStateVar $
      alloca $ \buf -> do
         glGetProgramiv (programID program) (marshalGetProgramPName p) buf
         peek1 f buf

foreign import unsafe "dynamic" dyn_glGetProgramiv :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (GLuint -> GLenum -> Ptr GLint -> IO ()) ; glGetProgramiv :: (GLuint -> GLenum -> Ptr GLint -> IO ()) ; glGetProgramiv = dyn_glGetProgramiv ptr_glGetProgramiv ; ptr_glGetProgramiv :: FunPtr a ; ptr_glGetProgramiv = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

--------------------------------------------------------------------------------

attribLocation :: Program -> String -> StateVar AttribLocation
attribLocation program name =
   makeStateVar (getAttribLocation program name)
                (\location -> bindAttribLocation program location name)

getAttribLocation :: Program -> String -> IO AttribLocation
getAttribLocation program name =
   withGLStringLen name $ \(buf,_) ->
      fmap (AttribLocation . fromIntegral) $
        glGetAttribLocation program buf

foreign import unsafe "dynamic" dyn_glGetAttribLocation :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (Program -> Ptr GLchar -> IO GLint) ; glGetAttribLocation :: (Program -> Ptr GLchar -> IO GLint) ; glGetAttribLocation = dyn_glGetAttribLocation ptr_glGetAttribLocation ; ptr_glGetAttribLocation :: FunPtr a ; ptr_glGetAttribLocation = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

bindAttribLocation :: Program -> AttribLocation -> String -> IO ()
bindAttribLocation program location name =
   withGLStringLen name $ \(buf,_) ->
      glBindAttribLocation program location buf

foreign import unsafe "dynamic" dyn_glBindAttribLocation :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (Program -> AttribLocation -> Ptr GLchar -> IO ()) ; glBindAttribLocation :: (Program -> AttribLocation -> Ptr GLchar -> IO ()) ; glBindAttribLocation = dyn_glBindAttribLocation ptr_glBindAttribLocation ; ptr_glBindAttribLocation :: FunPtr a ; ptr_glBindAttribLocation = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

--------------------------------------------------------------------------------

data VariableType =
     Float'
   | FloatVec2
   | FloatVec3
   | FloatVec4
   | FloatMat2
   | FloatMat3
   | FloatMat4
   | Int'
   | IntVec2
   | IntVec3
   | IntVec4
   | Bool
   | BoolVec2
   | BoolVec3
   | BoolVec4
   | Sampler1D
   | Sampler2D
   | Sampler3D
   | SamplerCube
   | Sampler1DShadow
   | Sampler2DShadow
   deriving ( Eq, Ord, Show )

unmarshalVariableType :: GLenum -> VariableType
unmarshalVariableType x
   | x == 0x1406 = Float'
   | x == 0x8B50 = FloatVec2
   | x == 0x8B51 = FloatVec3
   | x == 0x8B52 = FloatVec4
   | x == 0x8B5A = FloatMat2
   | x == 0x8B5B = FloatMat3
   | x == 0x8B5C = FloatMat4
   | x == 0x1404 = Int'
   | x == 0x8B53 = IntVec2
   | x == 0x8B54 = IntVec3
   | x == 0x8B55 = IntVec4
   | x == 0x8B56 = Bool
   | x == 0x8B57 = BoolVec2
   | x == 0x8B58 = BoolVec3
   | x == 0x8B59 = BoolVec4
   | x == 0x8B5D = Sampler1D
   | x == 0x8B5E = Sampler2D
   | x == 0x8B5F = Sampler3D
   | x == 0x8B60 = SamplerCube
   | x == 0x8B61 = Sampler1DShadow
   | x == 0x8B62 = Sampler2DShadow
   | otherwise = error ("unmarshalVariableType: illegal value " ++ show x)

--------------------------------------------------------------------------------

activeVars :: (Program -> GettableStateVar GLuint)
           -> (Program -> GettableStateVar GLsizei)
           -> (Program -> GLuint -> GLsizei -> Ptr GLsizei -> Ptr GLint -> Ptr GLenum -> Ptr GLchar -> IO ())
           -> Program -> GettableStateVar [(GLint,VariableType,String)]
activeVars numVars maxLength getter program =
   makeGettableStateVar $ do
      numActiveVars <- get (numVars program)
      maxLen <- get (maxLength program)
      allocaArray (fromIntegral maxLen) $ \nameBuf ->
         alloca $ \nameLengthBuf ->
            alloca $ \sizeBuf ->
               alloca $ \typeBuf ->
                  flip mapM [0 .. numActiveVars - 1] $ \i -> do
                    getter program i maxLen nameLengthBuf sizeBuf typeBuf nameBuf
                    l <- peek nameLengthBuf
                    s <- peek sizeBuf
                    t <- peek typeBuf
                    n <- peekGLstringLen (nameBuf, l)
                    return (s, unmarshalVariableType t, n)

activeAttribs :: Program -> GettableStateVar [(GLint,VariableType,String)]
activeAttribs = activeVars activeAttributes activeAttributeMaxLength glGetActiveAttrib

foreign import unsafe "dynamic" dyn_glGetActiveAttrib :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (Program -> GLuint -> GLsizei -> Ptr GLsizei -> Ptr GLint -> Ptr GLenum -> Ptr GLchar -> IO ()) ; glGetActiveAttrib :: (Program -> GLuint -> GLsizei -> Ptr GLsizei -> Ptr GLint -> Ptr GLenum -> Ptr GLchar -> IO ()) ; glGetActiveAttrib = dyn_glGetActiveAttrib ptr_glGetActiveAttrib ; ptr_glGetActiveAttrib :: FunPtr a ; ptr_glGetActiveAttrib = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

--------------------------------------------------------------------------------

newtype UniformLocation = UniformLocation GLint
   deriving ( Eq, Ord, Show )

uniformLocation :: Program -> String -> GettableStateVar UniformLocation
uniformLocation program name =
   makeGettableStateVar $
      withGLStringLen name $ \(buf,_) ->
         fmap UniformLocation $
            glGetUniformLocation program buf

foreign import unsafe "dynamic" dyn_glGetUniformLocation :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (Program -> Ptr GLchar -> IO GLint) ; glGetUniformLocation :: (Program -> Ptr GLchar -> IO GLint) ; glGetUniformLocation = dyn_glGetUniformLocation ptr_glGetUniformLocation ; ptr_glGetUniformLocation :: FunPtr a ; ptr_glGetUniformLocation = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

--------------------------------------------------------------------------------

activeUniforms :: Program -> GettableStateVar [(GLint,VariableType,String)]
activeUniforms = activeVars numActiveUniforms activeUniformMaxLength glGetActiveUniform

foreign import unsafe "dynamic" dyn_glGetActiveUniform :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (Program -> GLuint -> GLsizei -> Ptr GLsizei -> Ptr GLint -> Ptr GLenum -> Ptr GLchar -> IO ()) ; glGetActiveUniform :: (Program -> GLuint -> GLsizei -> Ptr GLsizei -> Ptr GLint -> Ptr GLenum -> Ptr GLchar -> IO ()) ; glGetActiveUniform = dyn_glGetActiveUniform ptr_glGetActiveUniform ; ptr_glGetActiveUniform :: FunPtr a ; ptr_glGetActiveUniform = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

--------------------------------------------------------------------------------

class Storable a => UniformComponent a where
   uniform1 :: UniformLocation -> a -> IO ()
   uniform2 :: UniformLocation -> a -> a -> IO ()
   uniform3 :: UniformLocation -> a -> a -> a -> IO ()
   uniform4 :: UniformLocation -> a -> a -> a -> a -> IO ()

   getUniform :: Storable (b a) => Program -> UniformLocation -> Ptr (b a) -> IO ()

   uniform1v :: UniformLocation -> GLsizei -> Ptr a -> IO ()
   uniform2v :: UniformLocation -> GLsizei -> Ptr a -> IO ()
   uniform3v :: UniformLocation -> GLsizei -> Ptr a -> IO ()
   uniform4v :: UniformLocation -> GLsizei -> Ptr a -> IO ()

--------------------------------------------------------------------------------

foreign import unsafe "dynamic" dyn_glUniform1i :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLint -> IO ()) ; glUniform1i :: (UniformLocation -> GLint -> IO ()) ; glUniform1i = dyn_glUniform1i ptr_glUniform1i ; ptr_glUniform1i :: FunPtr a ; ptr_glUniform1i = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glUniform2i :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLint -> GLint -> IO ()) ; glUniform2i :: (UniformLocation -> GLint -> GLint -> IO ()) ; glUniform2i = dyn_glUniform2i ptr_glUniform2i ; ptr_glUniform2i :: FunPtr a ; ptr_glUniform2i = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glUniform3i :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLint -> GLint -> GLint -> IO ()) ; glUniform3i :: (UniformLocation -> GLint -> GLint -> GLint -> IO ()) ; glUniform3i = dyn_glUniform3i ptr_glUniform3i ; ptr_glUniform3i :: FunPtr a ; ptr_glUniform3i = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glUniform4i :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLint -> GLint -> GLint -> GLint -> IO ()) ; glUniform4i :: (UniformLocation -> GLint -> GLint -> GLint -> GLint -> IO ()) ; glUniform4i = dyn_glUniform4i ptr_glUniform4i ; ptr_glUniform4i :: FunPtr a ; ptr_glUniform4i = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

foreign import unsafe "dynamic" dyn_glGetUniformiv :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (Program -> UniformLocation -> Ptr GLint -> IO ()) ; glGetUniformiv :: (Program -> UniformLocation -> Ptr GLint -> IO ()) ; glGetUniformiv = dyn_glGetUniformiv ptr_glGetUniformiv ; ptr_glGetUniformiv :: FunPtr a ; ptr_glGetUniformiv = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

foreign import unsafe "dynamic" dyn_glUniform1iv :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLsizei -> Ptr GLint -> IO ()) ; glUniform1iv :: (UniformLocation -> GLsizei -> Ptr GLint -> IO ()) ; glUniform1iv = dyn_glUniform1iv ptr_glUniform1iv ; ptr_glUniform1iv :: FunPtr a ; ptr_glUniform1iv = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glUniform2iv :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLsizei -> Ptr GLint -> IO ()) ; glUniform2iv :: (UniformLocation -> GLsizei -> Ptr GLint -> IO ()) ; glUniform2iv = dyn_glUniform2iv ptr_glUniform2iv ; ptr_glUniform2iv :: FunPtr a ; ptr_glUniform2iv = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glUniform3iv :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLsizei -> Ptr GLint -> IO ()) ; glUniform3iv :: (UniformLocation -> GLsizei -> Ptr GLint -> IO ()) ; glUniform3iv = dyn_glUniform3iv ptr_glUniform3iv ; ptr_glUniform3iv :: FunPtr a ; ptr_glUniform3iv = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glUniform4iv :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLsizei -> Ptr GLint -> IO ()) ; glUniform4iv :: (UniformLocation -> GLsizei -> Ptr GLint -> IO ()) ; glUniform4iv = dyn_glUniform4iv ptr_glUniform4iv ; ptr_glUniform4iv :: FunPtr a ; ptr_glUniform4iv = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

instance UniformComponent Int32 where
   uniform1 = glUniform1i
   uniform2 = glUniform2i
   uniform3 = glUniform3i
   uniform4 = glUniform4i

   getUniform program location = glGetUniformiv program location . castPtr

   uniform1v = glUniform1iv
   uniform2v = glUniform2iv
   uniform3v = glUniform3iv
   uniform4v = glUniform4iv

--------------------------------------------------------------------------------

foreign import unsafe "dynamic" dyn_glUniform1f :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLfloat -> IO ()) ; glUniform1f :: (UniformLocation -> GLfloat -> IO ()) ; glUniform1f = dyn_glUniform1f ptr_glUniform1f ; ptr_glUniform1f :: FunPtr a ; ptr_glUniform1f = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glUniform2f :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLfloat -> GLfloat -> IO ()) ; glUniform2f :: (UniformLocation -> GLfloat -> GLfloat -> IO ()) ; glUniform2f = dyn_glUniform2f ptr_glUniform2f ; ptr_glUniform2f :: FunPtr a ; ptr_glUniform2f = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glUniform3f :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLfloat -> GLfloat -> GLfloat -> IO ()) ; glUniform3f :: (UniformLocation -> GLfloat -> GLfloat -> GLfloat -> IO ()) ; glUniform3f = dyn_glUniform3f ptr_glUniform3f ; ptr_glUniform3f :: FunPtr a ; ptr_glUniform3f = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glUniform4f :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLfloat -> GLfloat -> GLfloat -> GLfloat -> IO ()) ; glUniform4f :: (UniformLocation -> GLfloat -> GLfloat -> GLfloat -> GLfloat -> IO ()) ; glUniform4f = dyn_glUniform4f ptr_glUniform4f ; ptr_glUniform4f :: FunPtr a ; ptr_glUniform4f = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

foreign import unsafe "dynamic" dyn_glGetUniformfv :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (Program -> UniformLocation -> Ptr GLfloat -> IO ()) ; glGetUniformfv :: (Program -> UniformLocation -> Ptr GLfloat -> IO ()) ; glGetUniformfv = dyn_glGetUniformfv ptr_glGetUniformfv ; ptr_glGetUniformfv :: FunPtr a ; ptr_glGetUniformfv = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

foreign import unsafe "dynamic" dyn_glUniform1fv :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLsizei -> Ptr GLfloat -> IO ()) ; glUniform1fv :: (UniformLocation -> GLsizei -> Ptr GLfloat -> IO ()) ; glUniform1fv = dyn_glUniform1fv ptr_glUniform1fv ; ptr_glUniform1fv :: FunPtr a ; ptr_glUniform1fv = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glUniform2fv :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLsizei -> Ptr GLfloat -> IO ()) ; glUniform2fv :: (UniformLocation -> GLsizei -> Ptr GLfloat -> IO ()) ; glUniform2fv = dyn_glUniform2fv ptr_glUniform2fv ; ptr_glUniform2fv :: FunPtr a ; ptr_glUniform2fv = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glUniform3fv :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLsizei -> Ptr GLfloat -> IO ()) ; glUniform3fv :: (UniformLocation -> GLsizei -> Ptr GLfloat -> IO ()) ; glUniform3fv = dyn_glUniform3fv ptr_glUniform3fv ; ptr_glUniform3fv :: FunPtr a ; ptr_glUniform3fv = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glUniform4fv :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLsizei -> Ptr GLfloat -> IO ()) ; glUniform4fv :: (UniformLocation -> GLsizei -> Ptr GLfloat -> IO ()) ; glUniform4fv = dyn_glUniform4fv ptr_glUniform4fv ; ptr_glUniform4fv :: FunPtr a ; ptr_glUniform4fv = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;

instance UniformComponent Float where
   uniform1 = glUniform1f
   uniform2 = glUniform2f
   uniform3 = glUniform3f
   uniform4 = glUniform4f

   getUniform program location = glGetUniformfv program location . castPtr

   uniform1v = glUniform1fv
   uniform2v = glUniform2fv
   uniform3v = glUniform3fv
   uniform4v = glUniform4fv

--------------------------------------------------------------------------------

foreign import unsafe "dynamic" dyn_glUniformMatrix2fv :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLsizei -> GLboolean -> Ptr GLfloat -> IO ()) ; glUniformMatrix2fv :: (UniformLocation -> GLsizei -> GLboolean -> Ptr GLfloat -> IO ()) ; glUniformMatrix2fv = dyn_glUniformMatrix2fv ptr_glUniformMatrix2fv ; ptr_glUniformMatrix2fv :: FunPtr a ; ptr_glUniformMatrix2fv = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glUniformMatrix3fv :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLsizei -> GLboolean -> Ptr GLfloat -> IO ()) ; glUniformMatrix3fv :: (UniformLocation -> GLsizei -> GLboolean -> Ptr GLfloat -> IO ()) ; glUniformMatrix3fv = dyn_glUniformMatrix3fv ptr_glUniformMatrix3fv ; ptr_glUniformMatrix3fv :: FunPtr a ; ptr_glUniformMatrix3fv = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glUniformMatrix4fv :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLsizei -> GLboolean -> Ptr GLfloat -> IO ()) ; glUniformMatrix4fv :: (UniformLocation -> GLsizei -> GLboolean -> Ptr GLfloat -> IO ()) ; glUniformMatrix4fv = dyn_glUniformMatrix4fv ptr_glUniformMatrix4fv ; ptr_glUniformMatrix4fv :: FunPtr a ; ptr_glUniformMatrix4fv = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.0") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glUniformMatrix2x3fv :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLsizei -> GLboolean -> Ptr GLfloat -> IO ()) ; glUniformMatrix2x3fv :: (UniformLocation -> GLsizei -> GLboolean -> Ptr GLfloat -> IO ()) ; glUniformMatrix2x3fv = dyn_glUniformMatrix2x3fv ptr_glUniformMatrix2x3fv ; ptr_glUniformMatrix2x3fv :: FunPtr a ; ptr_glUniformMatrix2x3fv = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.1") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glUniformMatrix3x2fv :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLsizei -> GLboolean -> Ptr GLfloat -> IO ()) ; glUniformMatrix3x2fv :: (UniformLocation -> GLsizei -> GLboolean -> Ptr GLfloat -> IO ()) ; glUniformMatrix3x2fv = dyn_glUniformMatrix3x2fv ptr_glUniformMatrix3x2fv ; ptr_glUniformMatrix3x2fv :: FunPtr a ; ptr_glUniformMatrix3x2fv = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.1") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glUniformMatrix2x4fv :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLsizei -> GLboolean -> Ptr GLfloat -> IO ()) ; glUniformMatrix2x4fv :: (UniformLocation -> GLsizei -> GLboolean -> Ptr GLfloat -> IO ()) ; glUniformMatrix2x4fv = dyn_glUniformMatrix2x4fv ptr_glUniformMatrix2x4fv ; ptr_glUniformMatrix2x4fv :: FunPtr a ; ptr_glUniformMatrix2x4fv = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.1") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glUniformMatrix4x2fv :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLsizei -> GLboolean -> Ptr GLfloat -> IO ()) ; glUniformMatrix4x2fv :: (UniformLocation -> GLsizei -> GLboolean -> Ptr GLfloat -> IO ()) ; glUniformMatrix4x2fv = dyn_glUniformMatrix4x2fv ptr_glUniformMatrix4x2fv ; ptr_glUniformMatrix4x2fv :: FunPtr a ; ptr_glUniformMatrix4x2fv = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.1") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glUniformMatrix3x4fv :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLsizei -> GLboolean -> Ptr GLfloat -> IO ()) ; glUniformMatrix3x4fv :: (UniformLocation -> GLsizei -> GLboolean -> Ptr GLfloat -> IO ()) ; glUniformMatrix3x4fv = dyn_glUniformMatrix3x4fv ptr_glUniformMatrix3x4fv ; ptr_glUniformMatrix3x4fv :: FunPtr a ; ptr_glUniformMatrix3x4fv = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.1") ("_entry")) ;
foreign import unsafe "dynamic" dyn_glUniformMatrix4x3fv :: Graphics.Rendering.OpenGL.GL.Extensions.Invoker (UniformLocation -> GLsizei -> GLboolean -> Ptr GLfloat -> IO ()) ; glUniformMatrix4x3fv :: (UniformLocation -> GLsizei -> GLboolean -> Ptr GLfloat -> IO ()) ; glUniformMatrix4x3fv = dyn_glUniformMatrix4x3fv ptr_glUniformMatrix4x3fv ; ptr_glUniformMatrix4x3fv :: FunPtr a ; ptr_glUniformMatrix4x3fv = unsafePerformIO (Graphics.Rendering.OpenGL.GL.Extensions.getProcAddress ("OpenGL 2.1") ("_entry")) ;

--------------------------------------------------------------------------------

class Uniform a where
   uniform :: UniformLocation -> StateVar a
   uniformv :: UniformLocation -> GLsizei -> Ptr a -> IO ()

maxComponentSize :: Int
maxComponentSize = sizeOf (undefined :: GLint) `max` sizeOf (undefined :: GLfloat)

maxNumComponents :: Int
maxNumComponents = 16

maxUniformBufferSize :: Int
maxUniformBufferSize = maxComponentSize * maxNumComponents

makeUniformVar :: (UniformComponent a, Storable (b a))
               => (UniformLocation -> b a -> IO ())
               -> UniformLocation -> StateVar (b a)
makeUniformVar setter location = makeStateVar getter (setter location)
   where getter = do
            program <- getCurrentProgram
            allocaBytes maxUniformBufferSize $ \buf -> do
            getUniform program location buf
            peek buf

instance UniformComponent a => Uniform (Vertex2 a) where
   uniform = makeUniformVar $ \location (Vertex2 x y) -> uniform2 location x y
   uniformv location count = uniform2v location count . (castPtr :: Ptr (Vertex2 b) -> Ptr b)

instance UniformComponent a => Uniform (Vertex3 a) where
   uniform = makeUniformVar $ \location (Vertex3 x y z) -> uniform3 location x y z
   uniformv location count = uniform3v location count . (castPtr :: Ptr (Vertex3 b) -> Ptr b)

instance UniformComponent a => Uniform (Vertex4 a) where
   uniform = makeUniformVar $ \location (Vertex4 x y z w) -> uniform4 location x y z w
   uniformv location count = uniform4v location count . (castPtr :: Ptr (Vertex4 b) -> Ptr b)

instance UniformComponent a => Uniform (TexCoord1 a) where
   uniform = makeUniformVar $ \location (TexCoord1 s) -> uniform1 location s
   uniformv location count = uniform1v location count . (castPtr :: Ptr (TexCoord1 b) -> Ptr b)

instance UniformComponent a => Uniform (TexCoord2 a) where
   uniform = makeUniformVar $ \location (TexCoord2 s t) -> uniform2 location s t
   uniformv location count = uniform2v location count . (castPtr :: Ptr (TexCoord2 b) -> Ptr b)

instance UniformComponent a => Uniform (TexCoord3 a) where
   uniform = makeUniformVar $ \location (TexCoord3 s t r) -> uniform3 location s t r
   uniformv location count = uniform3v location count . (castPtr :: Ptr (TexCoord3 b) -> Ptr b)

instance UniformComponent a => Uniform (TexCoord4 a) where
   uniform = makeUniformVar $ \location (TexCoord4 s t r q) -> uniform4 location s t r q
   uniformv location count = uniform4v location count . (castPtr :: Ptr (TexCoord4 b) -> Ptr b)

instance UniformComponent a => Uniform (Normal3 a) where
   uniform = makeUniformVar $ \location (Normal3 x y z) -> uniform3 location x y z
   uniformv location count = uniform3v location count . (castPtr :: Ptr (Normal3 b) -> Ptr b)

instance UniformComponent a => Uniform (FogCoord1 a) where
   uniform = makeUniformVar $ \location (FogCoord1 c) -> uniform1 location c
   uniformv location count = uniform1v location count . (castPtr :: Ptr (FogCoord1 b) -> Ptr b)

instance UniformComponent a => Uniform (Color3 a) where
   uniform = makeUniformVar $ \location (Color3 r g b) -> uniform3 location r g b
   uniformv location count = uniform3v location count . (castPtr :: Ptr (Color3 b) -> Ptr b)

instance UniformComponent a => Uniform (Color4 a) where
   uniform = makeUniformVar $ \location (Color4 r g b a) -> uniform4 location r g b a
   uniformv location count = uniform4v location count . (castPtr :: Ptr (Color4 b) -> Ptr b)

instance UniformComponent a => Uniform (Index1 a) where
   uniform = makeUniformVar $ \location (Index1 i) -> uniform1 location i
   uniformv location count = uniform1v location count . (castPtr :: Ptr (Index1 b) -> Ptr b)

--------------------------------------------------------------------------------

-- | Contains the number of hardware units that can be used to access texture
-- maps from the vertex processor. The minimum legal value is 0.

maxVertexTextureImageUnits :: GettableStateVar GLsizei
maxVertexTextureImageUnits = getLimit GetMaxVertexTextureImageUnits

-- | Contains the total number of hardware units that can be used to access
-- texture maps from the fragment processor. The minimum legal value is 2.

maxTextureImageUnits :: GettableStateVar GLsizei
maxTextureImageUnits = getLimit GetMaxTextureImageUnits

-- | Contains the total number of hardware units that can be used to access
-- texture maps from the vertex processor and the fragment processor combined.
-- Note: If the vertex shader and the fragment processing stage access the same
-- texture image unit, then that counts as using two texture image units. The
-- minimum legal value is 2.

maxCombinedTextureImageUnits :: GettableStateVar GLsizei
maxCombinedTextureImageUnits = getLimit GetMaxCombinedTextureImageUnits

-- | Contains the number of texture coordinate sets that are available. The
-- minimum legal value is 2.

maxTextureCoords :: GettableStateVar GLsizei
maxTextureCoords = getLimit GetMaxTextureCoords

-- | Contains the number of individual components (i.e., floating-point, integer
-- or boolean values) that are available for vertex shader uniform variables.
-- The minimum legal value is 512.
maxVertexUniformComponents :: GettableStateVar GLsizei
maxVertexUniformComponents = getLimit GetMaxVertexUniformComponents

-- | Contains the number of individual components (i.e., floating-point, integer
-- or boolean values) that are available for fragment shader uniform variables.
-- The minimum legal value is 64.

maxFragmentUniformComponents :: GettableStateVar GLsizei
maxFragmentUniformComponents = getLimit GetMaxFragmentUniformComponents

-- | Contains the number of active vertex attributes that are available. The
-- minimum legal value is 16.

maxVertexAttribs :: GettableStateVar GLsizei
maxVertexAttribs = getLimit GetMaxVertexAttribs

-- | Contains the number of individual floating-point values available for
-- varying variables. The minimum legal value is 32.

maxVaryingFloats :: GettableStateVar GLsizei
maxVaryingFloats = getLimit GetMaxVaryingFloats

getLimit :: GetPName -> GettableStateVar GLsizei
getLimit = makeGettableStateVar . getSizei1 id
